<?php
/**
 * Agent Response Endpoint
 * Allows live agent to send messages to escalated chats
 */

header('Content-Type: application/json');
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/auth_middleware.php';
require_once __DIR__ . '/../includes/pusher_config.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

try {
    // Authenticate vendor
    $vendorId = AuthMiddleware::getVendorId();
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (empty($input['session_id']) || empty($input['message'])) {
        http_response_code(400);
        echo json_encode(['error' => 'session_id and message are required']);
        exit;
    }
    
    $sessionId = $input['session_id'];
    $message = trim($input['message']);
    
    $db = getDB();
    
    // Verify session belongs to vendor and is escalated
    $stmt = $db->prepare("
        SELECT s.chatbot_id, s.status
        FROM active_sessions s
        WHERE s.session_id = ? AND s.vendor_id = ?
    ");
    $stmt->execute([$sessionId, $vendorId]);
    $session = $stmt->fetch();
    
    if (!$session) {
        http_response_code(404);
        echo json_encode(['error' => 'Session not found']);
        exit;
    }
    
    if ($session['status'] !== 'escalated') {
        http_response_code(400);
        echo json_encode(['error' => 'Session is not escalated']);
        exit;
    }
    
    // Log agent message
    $stmt = $db->prepare("
        INSERT INTO chat_logs (vendor_id, chatbot_id, session_id, role, message)
        VALUES (?, ?, ?, 'agent', ?)
    ");
    $stmt->execute([$vendorId, $session['chatbot_id'], $sessionId, $message]);
    
    // Send via Pusher to widget
    try {
        $pusher = PusherConfig::getInstance();
        $pusher->trigger(
            "vendor-{$vendorId}",
            "chat-update",
            [
                'session_id' => $sessionId,
                'role' => 'agent',
                'message' => $message,
                'timestamp' => date('Y-m-d H:i:s')
            ]
        );
    } catch (Exception $e) {
        error_log("Pusher error: " . $e->getMessage());
    }
    
    http_response_code(200);
    echo json_encode([
        'success' => true,
        'message' => 'Message sent successfully'
    ]);
    
} catch (Exception $e) {
    error_log("Agent response error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}
