<?php
/**
 * Vendor Login Endpoint
 * Handles vendor authentication and JWT token generation
 */

header('Content-Type: application/json');
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/jwt_helper.php';

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

try {
    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    
    // Validate required fields
    if (empty($input['email']) || empty($input['password'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Email and password are required']);
        exit;
    }
    
    $email = filter_var(trim($input['email']), FILTER_SANITIZE_EMAIL);
    $password = $input['password'];
    
    // Validate email format
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid email format']);
        exit;
    }
    
    $db = getDB();
    
    // Fetch vendor by email
    $stmt = $db->prepare("
        SELECT id, business_name, email, password_hash, plan_type, status, trial_ends_at
        FROM vendors
        WHERE email = ?
    ");
    $stmt->execute([$email]);
    $vendor = $stmt->fetch();
    
    // Check if vendor exists and password is correct
    if (!$vendor || !password_verify($password, $vendor['password_hash'])) {
        http_response_code(401);
        echo json_encode(['error' => 'Invalid email or password']);
        exit;
    }
    
    // Check if account is suspended
    if ($vendor['status'] === 'suspended') {
        http_response_code(403);
        echo json_encode(['error' => 'Account is suspended. Please contact support.']);
        exit;
    }
    
    // Generate JWT token
    $token = JWTHelper::encode([
        'vendor_id' => $vendor['id'],
        'email' => $vendor['email'],
        'plan_type' => $vendor['plan_type'],
        'trial_ends_at' => $vendor['trial_ends_at']
    ]);
    
    // Set token as HTTP-only cookie
    JWTHelper::setTokenCookie($token);
    
    // Return success response
    http_response_code(200);
    echo json_encode([
        'success' => true,
        'message' => 'Login successful',
        'vendor' => [
            'id' => $vendor['id'],
            'business_name' => $vendor['business_name'],
            'email' => $vendor['email'],
            'plan_type' => $vendor['plan_type'],
            'status' => $vendor['status'],
            'trial_ends_at' => $vendor['trial_ends_at']
        ],
        'token' => $token
    ]);
    
} catch (PDOException $e) {
    error_log("Login error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'Database error occurred']);
} catch (Exception $e) {
    error_log("Login error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}
