<?php
/**
 * Update Chatbot Endpoint
 * Saves chatbot configuration
 */

header('Content-Type: application/json');
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/auth_middleware.php';

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

try {
    // Authenticate vendor
    $vendorId = AuthMiddleware::getVendorId();
    
    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    
    // Validate required fields
    if (empty($input['bot_name']) || empty($input['welcome_message']) || empty($input['system_prompt'])) {
        http_response_code(400);
        echo json_encode(['error' => 'All fields are required']);
        exit;
    }
    
    $botName = trim($input['bot_name']);
    $welcomeMessage = trim($input['welcome_message']);
    $systemPrompt = trim($input['system_prompt']);
    
    $db = getDB();
    
    // Check if chatbot exists
    $stmt = $db->prepare("SELECT id FROM chatbots WHERE vendor_id = ? LIMIT 1");
    $stmt->execute([$vendorId]);
    $chatbot = $stmt->fetch();
    
    if ($chatbot) {
        // Update existing chatbot
        $stmt = $db->prepare("
            UPDATE chatbots
            SET bot_name = ?, welcome_message = ?, system_prompt = ?, updated_at = CURRENT_TIMESTAMP
            WHERE id = ? AND vendor_id = ?
        ");
        $stmt->execute([$botName, $welcomeMessage, $systemPrompt, $chatbot['id'], $vendorId]);
    } else {
        // Create new chatbot
        $stmt = $db->prepare("
            INSERT INTO chatbots (vendor_id, bot_name, welcome_message, system_prompt)
            VALUES (?, ?, ?, ?)
        ");
        $stmt->execute([$vendorId, $botName, $welcomeMessage, $systemPrompt]);
    }
    
    http_response_code(200);
    echo json_encode([
        'success' => true,
        'message' => 'Chatbot settings saved successfully'
    ]);
    
} catch (PDOException $e) {
    error_log("Update chatbot error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'Database error occurred']);
} catch (Exception $e) {
    error_log("Update chatbot error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}
