<?php
/**
 * AI Service
 * Handles OpenAI and Qdrant integration for RAG (Retrieval-Augmented Generation)
 */

require_once __DIR__ . '/env_loader.php';
require_once __DIR__ . '/db.php';

use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;

class AIService {
    private static $openaiClient = null;
    private static $qdrantClient = null;
    private static $embeddingModel = 'text-embedding-3-small';
    private static $chatModel = 'gpt-4o-mini';
    private static $chunkSize = 1000; // Characters per chunk
    private static $chunkOverlap = 200; // Overlap between chunks
    
    /**
     * Get OpenAI HTTP client
     */
    private static function getOpenAIClient() {
        if (self::$openaiClient === null) {
            $apiKey = getenv('OPENAI_API_KEY');
            if (empty($apiKey)) {
                throw new Exception("OPENAI_API_KEY is not set in environment variables");
            }
            
            self::$openaiClient = new Client([
                'base_uri' => 'https://api.openai.com/v1/',
                'headers' => [
                    'Authorization' => 'Bearer ' . $apiKey,
                    'Content-Type' => 'application/json'
                ],
                'timeout' => 30
            ]);
        }
        return self::$openaiClient;
    }
    
    /**
     * Get Qdrant HTTP client
     */
    private static function getQdrantClient() {
        if (self::$qdrantClient === null) {
            $apiKey = getenv('QDRANT_API_KEY');
            $clusterUrl = getenv('QDRANT_CLUSTER_URL');
            $collectionName = getenv('QDRANT_COLLECTION_NAME') ?: 'chatbot_knowledge';
            
            if (empty($apiKey) || empty($clusterUrl)) {
                throw new Exception("QDRANT_API_KEY and QDRANT_CLUSTER_URL must be set");
            }
            
            self::$qdrantClient = [
                'client' => new Client([
                    'base_uri' => rtrim($clusterUrl, '/') . '/',
                    'headers' => [
                        'api-key' => $apiKey,
                        'Content-Type' => 'application/json'
                    ],
                    'timeout' => 30
                ]),
                'collection' => $collectionName
            ];
        }
        return self::$qdrantClient;
    }
    
    /**
     * Get embedding vector from OpenAI
     * 
     * @param string $text Text to embed
     * @return array Embedding vector
     */
    private static function getEmbedding($text) {
        $client = self::getOpenAIClient();
        
        try {
            $response = $client->post('embeddings', [
                'json' => [
                    'model' => self::$embeddingModel,
                    'input' => $text
                ]
            ]);
            
            $data = json_decode($response->getBody()->getContents(), true);
            
            if (!isset($data['data'][0]['embedding'])) {
                throw new Exception("Failed to get embedding from OpenAI");
            }
            
            return $data['data'][0]['embedding'];
        } catch (RequestException $e) {
            error_log("OpenAI embedding error: " . $e->getMessage());
            throw new Exception("Failed to generate embedding: " . $e->getMessage());
        }
    }
    
    /**
     * Split text into chunks for processing
     * 
     * @param string $text Text to chunk
     * @return array Array of text chunks
     */
    private static function chunkText($text) {
        $chunks = [];
        $length = strlen($text);
        $start = 0;
        
        while ($start < $length) {
            $end = min($start + self::$chunkSize, $length);
            
            // Try to break at sentence boundary
            if ($end < $length) {
                $lastPeriod = strrpos(substr($text, $start, $end - $start), '.');
                $lastNewline = strrpos(substr($text, $start, $end - $start), "\n");
                $breakPoint = max($lastPeriod, $lastNewline);
                
                if ($breakPoint !== false) {
                    $end = $start + $breakPoint + 1;
                }
            }
            
            $chunk = substr($text, $start, $end - $start);
            if (!empty(trim($chunk))) {
                $chunks[] = trim($chunk);
            }
            
            $start = $end - self::$chunkOverlap;
        }
        
        return $chunks;
    }
    
    /**
     * Ingest document into Qdrant and store in knowledge base
     * 
     * @param string $text Document text
     * @param int $vendorId Vendor ID
     * @param string $contentType Content type (faq/doc/text)
     * @return array Array of qdrant_point_ids created
     */
    public static function ingestDocument($text, $vendorId, $contentType = 'text') {
        if (empty(trim($text))) {
            throw new Exception("Text cannot be empty");
        }
        
        $db = getDB();
        $qdrant = self::getQdrantClient();
        $client = $qdrant['client'];
        $collection = $qdrant['collection'];
        
        // Ensure collection exists
        self::ensureCollectionExists($collection);
        
        // Split text into chunks
        $chunks = self::chunkText($text);
        $pointIds = [];
        
        // Process each chunk
        foreach ($chunks as $chunkIndex => $chunk) {
            try {
                // Get embedding
                $embedding = self::getEmbedding($chunk);
                
                // Generate unique point ID
                $pointId = uniqid("vendor_{$vendorId}_", true);
                
                // Upsert to Qdrant
                $response = $client->put("collections/{$collection}/points", [
                    'json' => [
                        'points' => [[
                            'id' => $pointId,
                            'vector' => $embedding,
                            'payload' => [
                                'vendor_id' => $vendorId,
                                'content' => $chunk,
                                'content_type' => $contentType,
                                'chunk_index' => $chunkIndex
                            ]
                        ]]
                    ]
                ]);
                
                if ($response->getStatusCode() === 200) {
                    $pointIds[] = $pointId;
                    
                    // Store in knowledge_base table
                    $stmt = $db->prepare("
                        INSERT INTO knowledge_base (vendor_id, content_type, raw_content, qdrant_point_id)
                        VALUES (?, ?, ?, ?)
                    ");
                    $stmt->execute([$vendorId, $contentType, $chunk, $pointId]);
                }
            } catch (Exception $e) {
                error_log("Error ingesting chunk: " . $e->getMessage());
                // Continue with next chunk
            }
        }
        
        return $pointIds;
    }
    
    /**
     * Ensure Qdrant collection exists
     * 
     * @param string $collectionName Collection name
     */
    private static function ensureCollectionExists($collectionName) {
        $qdrant = self::getQdrantClient();
        $client = $qdrant['client'];
        
        try {
            // Check if collection exists
            $response = $client->get("collections/{$collectionName}");
            // Collection exists, return
            return;
        } catch (RequestException $e) {
            if ($e->getResponse() && $e->getResponse()->getStatusCode() === 404) {
                // Collection doesn't exist, create it
                try {
                    // Get embedding dimension (1536 for text-embedding-3-small)
                    $embedding = self::getEmbedding("test");
                    $dimension = count($embedding);
                    
                    $client->put("collections/{$collectionName}", [
                        'json' => [
                            'vectors' => [
                                'size' => $dimension,
                                'distance' => 'Cosine'
                            ]
                        ]
                    ]);
                } catch (Exception $e) {
                    error_log("Failed to create Qdrant collection: " . $e->getMessage());
                    throw new Exception("Failed to create Qdrant collection");
                }
            } else {
                throw $e;
            }
        }
    }
    
    /**
     * Search Qdrant for relevant knowledge
     * 
     * @param array $queryVector Query embedding vector
     * @param int $vendorId Vendor ID for filtering
     * @param int $limit Number of results
     * @return array Array of relevant points with scores
     */
    private static function searchQdrant($queryVector, $vendorId, $limit = 3) {
        $qdrant = self::getQdrantClient();
        $client = $qdrant['client'];
        $collection = $qdrant['collection'];
        
        try {
            $response = $client->post("collections/{$collection}/points/search", [
                'json' => [
                    'vector' => $queryVector,
                    'limit' => $limit,
                    'filter' => [
                        'must' => [
                            [
                                'key' => 'vendor_id',
                                'match' => ['value' => $vendorId]
                            ]
                        ]
                    ],
                    'with_payload' => true
                ]
            ]);
            
            $data = json_decode($response->getBody()->getContents(), true);
            
            return $data['result'] ?? [];
        } catch (RequestException $e) {
            error_log("Qdrant search error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get chat response using RAG
     * 
     * @param string $userQuery User's query
     * @param int $vendorId Vendor ID
     * @param int $chatbotId Chatbot ID
     * @return array Response with message and confidence score
     */
    public static function getChatResponse($userQuery, $vendorId, $chatbotId) {
        $db = getDB();
        
        // Get chatbot configuration
        $stmt = $db->prepare("
            SELECT system_prompt, welcome_message
            FROM chatbots
            WHERE id = ? AND vendor_id = ?
        ");
        $stmt->execute([$chatbotId, $vendorId]);
        $chatbot = $stmt->fetch();
        
        if (!$chatbot) {
            throw new Exception("Chatbot not found");
        }
        
        $systemPrompt = $chatbot['system_prompt'] ?? 'You are a helpful assistant. Answer questions based on the provided context.';
        
        // Get embedding for user query
        $queryVector = self::getEmbedding($userQuery);
        
        // Search Qdrant for relevant context
        $relevantPoints = self::searchQdrant($queryVector, $vendorId, 3);
        
        // Build context from retrieved knowledge
        $context = "";
        $confidence = 0.0;
        
        if (!empty($relevantPoints)) {
            $contextParts = [];
            foreach ($relevantPoints as $point) {
                if (isset($point['payload']['content'])) {
                    $contextParts[] = $point['payload']['content'];
                }
                // Use the highest score as confidence
                if (isset($point['score']) && $point['score'] > $confidence) {
                    $confidence = $point['score'];
                }
            }
            $context = implode("\n\n", $contextParts);
        }
        
        // Build prompt for OpenAI
        $messages = [
            [
                'role' => 'system',
                'content' => $systemPrompt
            ]
        ];
        
        if (!empty($context)) {
            $messages[] = [
                'role' => 'system',
                'content' => "Context from knowledge base:\n\n" . $context
            ];
        }
        
        $messages[] = [
            'role' => 'user',
            'content' => $userQuery
        ];
        
        // Call OpenAI API
        $client = self::getOpenAIClient();
        
        try {
            $response = $client->post('chat/completions', [
                'json' => [
                    'model' => self::$chatModel,
                    'messages' => $messages,
                    'temperature' => 0.7,
                    'max_tokens' => 500
                ]
            ]);
            
            $data = json_decode($response->getBody()->getContents(), true);
            
            if (!isset($data['choices'][0]['message']['content'])) {
                throw new Exception("Failed to get response from OpenAI");
            }
            
            $botResponse = $data['choices'][0]['message']['content'];
            
            // Log conversation
            $sessionId = $_POST['session_id'] ?? $_GET['session_id'] ?? uniqid('session_', true);
            
            // Log user message
            $stmt = $db->prepare("
                INSERT INTO chat_logs (vendor_id, chatbot_id, session_id, role, message)
                VALUES (?, ?, ?, 'user', ?)
            ");
            $stmt->execute([$vendorId, $chatbotId, $sessionId, $userQuery]);
            
            // Log bot response
            $stmt = $db->prepare("
                INSERT INTO chat_logs (vendor_id, chatbot_id, session_id, role, message)
                VALUES (?, ?, ?, 'bot', ?)
            ");
            $stmt->execute([$vendorId, $chatbotId, $sessionId, $botResponse]);
            
            return [
                'message' => $botResponse,
                'confidence' => $confidence,
                'session_id' => $sessionId
            ];
        } catch (RequestException $e) {
            error_log("OpenAI chat error: " . $e->getMessage());
            throw new Exception("Failed to generate chat response: " . $e->getMessage());
        }
    }
}
