<?php
/**
 * Authentication Middleware
 * Reusable JWT validation and vendor_id extraction
 */

require_once __DIR__ . '/jwt_helper.php';

class AuthMiddleware {
    /**
     * Validate JWT token and return vendor data
     * 
     * @return array Vendor data (vendor_id, email, plan_type, trial_ends_at)
     * @throws Exception If authentication fails
     */
    public static function authenticate() {
        $token = JWTHelper::getTokenFromRequest();
        
        if (!$token) {
            http_response_code(401);
            header('Content-Type: application/json');
            echo json_encode(['error' => 'Authentication required']);
            exit;
        }
        
        try {
            $vendorData = JWTHelper::decode($token);
            
            // Check if trial has expired
            if (!empty($vendorData['trial_ends_at'])) {
                $trialEndsAt = strtotime($vendorData['trial_ends_at']);
                if ($trialEndsAt < time()) {
                    // Trial expired - check if they've upgraded
                    require_once __DIR__ . '/db.php';
                    $db = getDB();
                    $stmt = $db->prepare("SELECT plan_type, status FROM vendors WHERE id = ?");
                    $stmt->execute([$vendorData['vendor_id']]);
                    $vendor = $stmt->fetch();
                    
                    if ($vendor && $vendor['plan_type'] === 'starter' && $vendor['status'] === 'trial') {
                        http_response_code(403);
                        header('Content-Type: application/json');
                        echo json_encode([
                            'error' => 'Trial period has expired',
                            'trial_ends_at' => $vendorData['trial_ends_at']
                        ]);
                        exit;
                    }
                }
            }
            
            return $vendorData;
        } catch (Exception $e) {
            http_response_code(401);
            header('Content-Type: application/json');
            echo json_encode(['error' => 'Invalid or expired token: ' . $e->getMessage()]);
            exit;
        }
    }
    
    /**
     * Get vendor ID from authenticated token
     * 
     * @return int Vendor ID
     */
    public static function getVendorId() {
        $vendorData = self::authenticate();
        return (int) $vendorData['vendor_id'];
    }
    
    /**
     * Check if vendor has Pro plan
     * 
     * @return bool True if Pro plan, false otherwise
     */
    public static function isProPlan() {
        $vendorData = self::authenticate();
        return $vendorData['plan_type'] === 'pro';
    }
    
    /**
     * Require Pro plan - exit with 403 if not Pro
     */
    public static function requireProPlan() {
        if (!self::isProPlan()) {
            http_response_code(403);
            header('Content-Type: application/json');
            echo json_encode(['error' => 'Pro plan required for this feature']);
            exit;
        }
    }
}
