<?php
/**
 * JWT Helper Functions
 * Handles JWT token encoding and decoding using firebase/php-jwt
 */

require_once __DIR__ . '/env_loader.php';

use Firebase\JWT\JWT;
use Firebase\JWT\Key;
use Firebase\JWT\ExpiredException;
use Firebase\JWT\SignatureInvalidException;

class JWTHelper {
    private static $secretKey;
    private static $algorithm = 'HS256';
    private static $expirationTime = 86400; // 24 hours in seconds
    
    private static function getSecretKey() {
        if (self::$secretKey === null) {
            self::$secretKey = getenv('JWT_SECRET_KEY');
            if (empty(self::$secretKey)) {
                throw new Exception("JWT_SECRET_KEY is not set in environment variables");
            }
        }
        return self::$secretKey;
    }
    
    /**
     * Encode JWT token with vendor data
     * 
     * @param array $payload Vendor data (vendor_id, email, plan_type, trial_ends_at)
     * @return string Encoded JWT token
     */
    public static function encode($payload) {
        $secretKey = self::getSecretKey();
        
        $tokenPayload = [
            'iss' => getenv('APP_URL') ?: 'http://localhost', // Issuer
            'iat' => time(), // Issued at
            'exp' => time() + self::$expirationTime, // Expiration
            'data' => [
                'vendor_id' => $payload['vendor_id'],
                'email' => $payload['email'],
                'plan_type' => $payload['plan_type'],
                'trial_ends_at' => $payload['trial_ends_at'] ?? null
            ]
        ];
        
        return JWT::encode($tokenPayload, $secretKey, self::$algorithm);
    }
    
    /**
     * Decode and validate JWT token
     * 
     * @param string $token JWT token string
     * @return array Decoded token data
     * @throws Exception If token is invalid or expired
     */
    public static function decode($token) {
        $secretKey = self::getSecretKey();
        
        try {
            $decoded = JWT::decode($token, new Key($secretKey, self::$algorithm));
            return (array) $decoded->data;
        } catch (ExpiredException $e) {
            throw new Exception("Token has expired");
        } catch (SignatureInvalidException $e) {
            throw new Exception("Invalid token signature");
        } catch (Exception $e) {
            throw new Exception("Invalid token: " . $e->getMessage());
        }
    }
    
    /**
     * Get JWT token from request headers or cookies
     * 
     * @return string|null JWT token or null if not found
     */
    public static function getTokenFromRequest() {
        // Check Authorization header first
        $headers = getallheaders();
        if (isset($headers['Authorization'])) {
            $authHeader = $headers['Authorization'];
            if (preg_match('/Bearer\s+(.*)$/i', $authHeader, $matches)) {
                return $matches[1];
            }
        }
        
        // Check cookie
        if (isset($_COOKIE['auth_token'])) {
            return $_COOKIE['auth_token'];
        }
        
        return null;
    }
    
    /**
     * Set JWT token as HTTP-only cookie
     * 
     * @param string $token JWT token
     * @param int $expiration Expiration time in seconds
     */
    public static function setTokenCookie($token, $expiration = null) {
        if ($expiration === null) {
            $expiration = time() + self::$expirationTime;
        }
        
        setcookie(
            'auth_token',
            $token,
            $expiration,
            '/',
            '',
            isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on',
            true // HTTP-only
        );
    }
    
    /**
     * Clear JWT token cookie
     */
    public static function clearTokenCookie() {
        setcookie('auth_token', '', time() - 3600, '/');
    }
}
