<?php
/**
 * Active Chats Page
 * Live agent dashboard for managing escalated chats
 */

session_start();
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/auth_middleware.php';
require_once __DIR__ . '/../includes/pusher_config.php';

// Authenticate user
$vendorData = AuthMiddleware::authenticate();
$vendorId = $vendorData['vendor_id'];

// Check if Pro plan
if ($vendorData['plan_type'] !== 'pro') {
    header('Location: index.php');
    exit;
}

// Get Pusher config for client
$pusherConfig = PusherConfig::getClientConfig();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Active Chats - AI Chatbot SaaS</title>
    <script src="https://js.pusher.com/8.2.0/pusher.min.js"></script>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            background: #f5f5f5;
            color: #333;
        }
        .header {
            background: #fff;
            border-bottom: 1px solid #e0e0e0;
            padding: 1rem 2rem;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        .header h1 {
            font-size: 1.5rem;
            color: #2563eb;
        }
        .nav-link {
            color: #2563eb;
            text-decoration: none;
            margin-right: 1rem;
        }
        .container {
            max-width: 1400px;
            margin: 2rem auto;
            padding: 0 2rem;
        }
        .chats-grid {
            display: grid;
            grid-template-columns: 350px 1fr;
            gap: 2rem;
        }
        .chats-list {
            background: #fff;
            border-radius: 8px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
            overflow: hidden;
        }
        .chats-list-header {
            padding: 1rem;
            background: #2563eb;
            color: white;
            font-weight: 600;
        }
        .chat-item {
            padding: 1rem;
            border-bottom: 1px solid #e5e7eb;
            cursor: pointer;
            transition: background 0.2s;
        }
        .chat-item:hover {
            background: #f9fafb;
        }
        .chat-item.active {
            background: #eff6ff;
            border-left: 3px solid #2563eb;
        }
        .chat-item-header {
            display: flex;
            justify-content: space-between;
            margin-bottom: 0.5rem;
        }
        .chat-item-name {
            font-weight: 600;
        }
        .chat-item-status {
            padding: 0.25rem 0.5rem;
            border-radius: 12px;
            font-size: 0.75rem;
            font-weight: 600;
        }
        .status-escalated {
            background: #fee2e2;
            color: #991b1b;
        }
        .chat-item-preview {
            font-size: 0.875rem;
            color: #6b7280;
            overflow: hidden;
            text-overflow: ellipsis;
            white-space: nowrap;
        }
        .chat-window {
            background: #fff;
            border-radius: 8px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
            display: flex;
            flex-direction: column;
            height: 700px;
        }
        .chat-window-header {
            padding: 1rem;
            border-bottom: 1px solid #e5e7eb;
            background: #f9fafb;
        }
        .chat-messages {
            flex: 1;
            overflow-y: auto;
            padding: 1rem;
            background: #f9fafb;
        }
        .message {
            margin-bottom: 1rem;
            display: flex;
            flex-direction: column;
        }
        .message.user {
            align-items: flex-end;
        }
        .message.agent {
            align-items: flex-start;
        }
        .message-bubble {
            max-width: 70%;
            padding: 0.75rem 1rem;
            border-radius: 12px;
            word-wrap: break-word;
        }
        .message.user .message-bubble {
            background: #2563eb;
            color: white;
        }
        .message.agent .message-bubble {
            background: #10b981;
            color: white;
        }
        .message-time {
            font-size: 0.75rem;
            color: #6b7280;
            margin-top: 0.25rem;
        }
        .chat-input-area {
            padding: 1rem;
            border-top: 1px solid #e5e7eb;
        }
        .chat-input-form {
            display: flex;
            gap: 0.5rem;
        }
        .chat-input {
            flex: 1;
            padding: 0.75rem;
            border: 1px solid #d1d5db;
            border-radius: 6px;
            font-size: 1rem;
        }
        .chat-send {
            padding: 0.75rem 1.5rem;
            background: #2563eb;
            color: white;
            border: none;
            border-radius: 6px;
            font-weight: 600;
            cursor: pointer;
        }
        .empty-state {
            text-align: center;
            padding: 4rem 2rem;
            color: #6b7280;
        }
    </style>
</head>
<body>
    <div class="header">
        <h1>AI Chatbot SaaS</h1>
        <div>
            <a href="index.php" class="nav-link">← Dashboard</a>
        </div>
    </div>

    <div class="container">
        <div class="chats-grid">
            <div class="chats-list">
                <div class="chats-list-header">Active Chats</div>
                <div id="chats-list-container"></div>
            </div>
            
            <div class="chat-window">
                <div class="chat-window-header">
                    <div id="chat-header-info">Select a chat to view messages</div>
                </div>
                <div class="chat-messages" id="chat-messages">
                    <div class="empty-state">Select a chat from the list to start responding</div>
                </div>
                <div class="chat-input-area" id="chat-input-area" style="display: none;">
                    <form id="chat-input-form">
                        <div class="chat-input-form">
                            <input 
                                type="text" 
                                id="chat-input" 
                                class="chat-input" 
                                placeholder="Type your response..."
                                autocomplete="off"
                            >
                            <button type="submit" class="chat-send">Send</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <script>
        const vendorId = <?php echo $vendorId; ?>;
        const pusherKey = '<?php echo $pusherConfig['key']; ?>';
        const pusherCluster = '<?php echo $pusherConfig['cluster']; ?>';
        let currentSessionId = null;
        let pusher = null;
        let channel = null;

        // Initialize Pusher
        if (pusherKey) {
            pusher = new Pusher(pusherKey, { cluster: pusherCluster });
            channel = pusher.subscribe('vendor-' + vendorId);
            
            channel.bind('chat-escalated', function(data) {
                loadChats();
            });
            
            channel.bind('chat-update', function(data) {
                if (currentSessionId === data.session_id) {
                    if (data.role === 'user') {
                        addMessage('user', data.message);
                    }
                }
                loadChats();
            });
        }

        // Load active chats
        async function loadChats() {
            try {
                const response = await fetch('../api/get_active_chats.php');
                const data = await response.json();
                
                if (data.success) {
                    renderChatsList(data.sessions);
                }
            } catch (error) {
                console.error('Error loading chats:', error);
            }
        }

        function renderChatsList(sessions) {
            const container = document.getElementById('chats-list-container');
            const escalatedSessions = sessions.filter(s => s.status === 'escalated');
            
            if (escalatedSessions.length === 0) {
                container.innerHTML = '<div class="empty-state" style="padding: 2rem;">No escalated chats</div>';
                return;
            }
            
            container.innerHTML = escalatedSessions.map(session => {
                const lastMessage = session.recent_messages && session.recent_messages.length > 0 
                    ? session.recent_messages[session.recent_messages.length - 1].message 
                    : 'No messages yet';
                return `
                    <div class="chat-item ${currentSessionId === session.session_id ? 'active' : ''}" 
                         onclick="selectChat('${session.session_id}', '${session.bot_name}')">
                        <div class="chat-item-header">
                            <span class="chat-item-name">${session.bot_name}</span>
                            <span class="chat-item-status status-escalated">Escalated</span>
                        </div>
                        <div class="chat-item-preview">${lastMessage.substring(0, 50)}...</div>
                    </div>
                `;
            }).join('');
        }

        function selectChat(sessionId, botName) {
            currentSessionId = sessionId;
            document.getElementById('chat-header-info').textContent = botName;
            document.getElementById('chat-input-area').style.display = 'block';
            loadChatMessages(sessionId);
            loadChats(); // Refresh list to show active state
        }

        async function loadChatMessages(sessionId) {
            try {
                const response = await fetch('../api/get_active_chats.php');
                const data = await response.json();
                
                if (data.success) {
                    const session = data.sessions.find(s => s.session_id === sessionId);
                    if (session && session.recent_messages) {
                        renderMessages(session.recent_messages);
                    }
                }
            } catch (error) {
                console.error('Error loading messages:', error);
            }
        }

        function renderMessages(messages) {
            const container = document.getElementById('chat-messages');
            container.innerHTML = messages.map(msg => `
                <div class="message ${msg.role}">
                    <div class="message-bubble">${escapeHtml(msg.message)}</div>
                    <div class="message-time">${new Date(msg.created_at).toLocaleString()}</div>
                </div>
            `).join('');
            container.scrollTop = container.scrollHeight;
        }

        function addMessage(role, message) {
            const container = document.getElementById('chat-messages');
            const messageDiv = document.createElement('div');
            messageDiv.className = `message ${role}`;
            messageDiv.innerHTML = `
                <div class="message-bubble">${escapeHtml(message)}</div>
                <div class="message-time">${new Date().toLocaleString()}</div>
            `;
            container.appendChild(messageDiv);
            container.scrollTop = container.scrollHeight;
        }

        document.getElementById('chat-input-form').addEventListener('submit', async function(e) {
            e.preventDefault();
            
            if (!currentSessionId) return;
            
            const input = document.getElementById('chat-input');
            const message = input.value.trim();
            
            if (!message) return;
            
            try {
                const response = await fetch('../api/agent_response.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({
                        session_id: currentSessionId,
                        message: message
                    })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    addMessage('agent', message);
                    input.value = '';
                } else {
                    alert('Error: ' + (data.error || 'Failed to send message'));
                }
            } catch (error) {
                alert('Network error: ' + error.message);
            }
        });

        function escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }

        // Load chats on page load
        loadChats();
        setInterval(loadChats, 5000); // Refresh every 5 seconds
    </script>
</body>
</html>
