<?php
/**
 * Chatbot Settings Page
 * Configure chatbot name, welcome message, and system prompt
 */

session_start();
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/auth_middleware.php';

// Authenticate user
$vendorData = AuthMiddleware::authenticate();
$vendorId = $vendorData['vendor_id'];

// Get chatbot configuration
$db = getDB();
$stmt = $db->prepare("
    SELECT id, bot_name, welcome_message, system_prompt, live_agent_status
    FROM chatbots
    WHERE vendor_id = ?
    LIMIT 1
");
$stmt->execute([$vendorId]);
$chatbot = $stmt->fetch();

if (!$chatbot) {
    // Create default chatbot if none exists
    $stmt = $db->prepare("
        INSERT INTO chatbots (vendor_id, bot_name, welcome_message, system_prompt)
        VALUES (?, 'My Chatbot', 'Hello! How can I help you today?', 'You are a helpful assistant. Answer questions based on the provided context.')
    ");
    $stmt->execute([$vendorId]);
    $chatbotId = $db->lastInsertId();
    
    $stmt = $db->prepare("
        SELECT id, bot_name, welcome_message, system_prompt, live_agent_status
        FROM chatbots
        WHERE id = ?
    ");
    $stmt->execute([$chatbotId]);
    $chatbot = $stmt->fetch();
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Chatbot Settings - AI Chatbot SaaS</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            background: #f5f5f5;
            color: #333;
        }
        .header {
            background: #fff;
            border-bottom: 1px solid #e0e0e0;
            padding: 1rem 2rem;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        .header h1 {
            font-size: 1.5rem;
            color: #2563eb;
        }
        .nav-link {
            color: #2563eb;
            text-decoration: none;
            margin-right: 1rem;
        }
        .container {
            max-width: 800px;
            margin: 2rem auto;
            padding: 0 2rem;
        }
        .card {
            background: #fff;
            padding: 2rem;
            border-radius: 8px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
        }
        .card h2 {
            margin-bottom: 1.5rem;
            font-size: 1.5rem;
        }
        .form-group {
            margin-bottom: 1.5rem;
        }
        .form-group label {
            display: block;
            margin-bottom: 0.5rem;
            font-weight: 600;
            color: #374151;
        }
        .form-group input,
        .form-group textarea {
            width: 100%;
            padding: 0.75rem;
            border: 1px solid #d1d5db;
            border-radius: 6px;
            font-size: 1rem;
            font-family: inherit;
        }
        .form-group textarea {
            min-height: 120px;
            resize: vertical;
        }
        .form-group input:focus,
        .form-group textarea:focus {
            outline: none;
            border-color: #2563eb;
            box-shadow: 0 0 0 3px rgba(37, 99, 235, 0.1);
        }
        .form-group small {
            display: block;
            margin-top: 0.25rem;
            color: #6b7280;
            font-size: 0.875rem;
        }
        .btn {
            padding: 0.75rem 1.5rem;
            border: none;
            border-radius: 6px;
            font-size: 1rem;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.2s;
        }
        .btn-primary {
            background: #2563eb;
            color: white;
        }
        .btn-primary:hover {
            background: #1d4ed8;
        }
        .alert {
            padding: 1rem;
            border-radius: 6px;
            margin-bottom: 1rem;
        }
        .alert-success {
            background: #d1fae5;
            border: 1px solid #10b981;
            color: #065f46;
        }
        .alert-error {
            background: #fee2e2;
            border: 1px solid #ef4444;
            color: #991b1b;
        }
    </style>
</head>
<body>
    <div class="header">
        <h1>AI Chatbot SaaS</h1>
        <div>
            <a href="index.php" class="nav-link">← Dashboard</a>
        </div>
    </div>

    <div class="container">
        <div class="card">
            <h2>Chatbot Settings</h2>
            <div id="alert-container"></div>
            
            <form id="settings-form">
                <div class="form-group">
                    <label for="bot_name">Bot Name</label>
                    <input 
                        type="text" 
                        id="bot_name" 
                        name="bot_name" 
                        value="<?php echo htmlspecialchars($chatbot['bot_name']); ?>"
                        required
                    >
                    <small>This name will be displayed in the chat widget</small>
                </div>
                
                <div class="form-group">
                    <label for="welcome_message">Welcome Message</label>
                    <textarea 
                        id="welcome_message" 
                        name="welcome_message" 
                        required
                    ><?php echo htmlspecialchars($chatbot['welcome_message']); ?></textarea>
                    <small>The first message users see when they open the chat</small>
                </div>
                
                <div class="form-group">
                    <label for="system_prompt">System Prompt</label>
                    <textarea 
                        id="system_prompt" 
                        name="system_prompt" 
                        required
                    ><?php echo htmlspecialchars($chatbot['system_prompt']); ?></textarea>
                    <small>Instructions for how the AI should behave and respond to users</small>
                </div>
                
                <button type="submit" class="btn btn-primary">Save Settings</button>
            </form>
        </div>
    </div>

    <script>
        document.getElementById('settings-form').addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const formData = {
                bot_name: document.getElementById('bot_name').value,
                welcome_message: document.getElementById('welcome_message').value,
                system_prompt: document.getElementById('system_prompt').value
            };
            
            const alertContainer = document.getElementById('alert-container');
            alertContainer.innerHTML = '';
            
            try {
                const response = await fetch('../api/update_chatbot.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify(formData)
                });
                
                const data = await response.json();
                
                if (response.ok && data.success) {
                    alertContainer.innerHTML = '<div class="alert alert-success">✓ ' + data.message + '</div>';
                } else {
                    alertContainer.innerHTML = '<div class="alert alert-error">✗ ' + (data.error || 'Failed to save settings') + '</div>';
                }
            } catch (error) {
                alertContainer.innerHTML = '<div class="alert alert-error">✗ Network error: ' + error.message + '</div>';
            }
        });
    </script>
</body>
</html>
