<?php
/**
 * Vendor Portal Dashboard
 * Main dashboard page with authentication check
 */

session_start();
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/auth_middleware.php';

// Authenticate user
$vendorData = AuthMiddleware::authenticate();
$vendorId = $vendorData['vendor_id'];

// Get vendor details from database
$db = getDB();
$stmt = $db->prepare("
    SELECT id, business_name, email, plan_type, status, trial_ends_at, created_at
    FROM vendors
    WHERE id = ?
");
$stmt->execute([$vendorId]);
$vendor = $stmt->fetch();

// Get chatbot count
$stmt = $db->prepare("SELECT COUNT(*) as count FROM chatbots WHERE vendor_id = ?");
$stmt->execute([$vendorId]);
$chatbotCount = $stmt->fetch()['count'];

// Get knowledge base count
$stmt = $db->prepare("SELECT COUNT(*) as count FROM knowledge_base WHERE vendor_id = ?");
$stmt->execute([$vendorId]);
$knowledgeCount = $stmt->fetch()['count'];

// Get active sessions count
$stmt = $db->prepare("SELECT COUNT(*) as count FROM active_sessions WHERE vendor_id = ? AND status IN ('active', 'escalated')");
$stmt->execute([$vendorId]);
$activeSessionsCount = $stmt->fetch()['count'];

// Check trial status
$trialExpired = false;
if (!empty($vendor['trial_ends_at'])) {
    $trialEndsAt = strtotime($vendor['trial_ends_at']);
    $trialExpired = $trialEndsAt < time() && $vendor['plan_type'] === 'starter';
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - AI Chatbot SaaS</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            background: #f5f5f5;
            color: #333;
        }
        .header {
            background: #fff;
            border-bottom: 1px solid #e0e0e0;
            padding: 1rem 2rem;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        .header h1 {
            font-size: 1.5rem;
            color: #2563eb;
        }
        .user-info {
            display: flex;
            align-items: center;
            gap: 1rem;
        }
        .plan-badge {
            padding: 0.25rem 0.75rem;
            border-radius: 12px;
            font-size: 0.875rem;
            font-weight: 600;
            text-transform: uppercase;
        }
        .plan-starter {
            background: #fef3c7;
            color: #92400e;
        }
        .plan-pro {
            background: #dbeafe;
            color: #1e40af;
        }
        .container {
            max-width: 1200px;
            margin: 2rem auto;
            padding: 0 2rem;
        }
        .alert {
            padding: 1rem;
            border-radius: 8px;
            margin-bottom: 2rem;
        }
        .alert-warning {
            background: #fef3c7;
            border: 1px solid #fbbf24;
            color: #92400e;
        }
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }
        .stat-card {
            background: #fff;
            padding: 1.5rem;
            border-radius: 8px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
        }
        .stat-card h3 {
            font-size: 0.875rem;
            color: #6b7280;
            margin-bottom: 0.5rem;
            text-transform: uppercase;
            letter-spacing: 0.05em;
        }
        .stat-card .value {
            font-size: 2rem;
            font-weight: 700;
            color: #2563eb;
        }
        .nav-menu {
            background: #fff;
            padding: 1.5rem;
            border-radius: 8px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
        }
        .nav-menu h2 {
            margin-bottom: 1rem;
            font-size: 1.25rem;
        }
        .nav-links {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
        }
        .nav-link {
            display: block;
            padding: 1rem;
            background: #f9fafb;
            border: 1px solid #e5e7eb;
            border-radius: 6px;
            text-decoration: none;
            color: #374151;
            transition: all 0.2s;
        }
        .nav-link:hover {
            background: #f3f4f6;
            border-color: #2563eb;
            color: #2563eb;
        }
        .nav-link strong {
            display: block;
            margin-bottom: 0.25rem;
        }
        .nav-link span {
            font-size: 0.875rem;
            color: #6b7280;
        }
        .logout-btn {
            padding: 0.5rem 1rem;
            background: #ef4444;
            color: white;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 0.875rem;
            text-decoration: none;
            display: inline-block;
        }
        .logout-btn:hover {
            background: #dc2626;
        }
    </style>
</head>
<body>
    <div class="header">
        <h1>AI Chatbot SaaS</h1>
        <div class="user-info">
            <span><?php echo htmlspecialchars($vendor['business_name']); ?></span>
            <span class="plan-badge plan-<?php echo $vendor['plan_type']; ?>">
                <?php echo ucfirst($vendor['plan_type']); ?>
            </span>
            <a href="../api/logout.php" class="logout-btn" onclick="return confirm('Are you sure you want to logout?')">Logout</a>
        </div>
    </div>

    <div class="container">
        <?php if ($trialExpired): ?>
        <div class="alert alert-warning">
            <strong>⚠️ Trial Expired:</strong> Your 30-day free trial has ended. Upgrade to Pro to continue using the service.
        </div>
        <?php elseif (!empty($vendor['trial_ends_at'])): ?>
        <div class="alert alert-warning">
            <strong>⏰ Trial Period:</strong> Your free trial ends on <?php echo date('F j, Y', strtotime($vendor['trial_ends_at'])); ?>
        </div>
        <?php endif; ?>

        <div class="stats-grid">
            <div class="stat-card">
                <h3>Chatbots</h3>
                <div class="value"><?php echo $chatbotCount; ?></div>
            </div>
            <div class="stat-card">
                <h3>Knowledge Entries</h3>
                <div class="value"><?php echo $knowledgeCount; ?></div>
            </div>
            <div class="stat-card">
                <h3>Active Sessions</h3>
                <div class="value"><?php echo $activeSessionsCount; ?></div>
            </div>
        </div>

        <div class="nav-menu">
            <h2>Quick Actions</h2>
            <div class="nav-links">
                <a href="chatbot_settings.php" class="nav-link">
                    <strong>⚙️ Chatbot Settings</strong>
                    <span>Configure your chatbot's name, welcome message, and system prompt</span>
                </a>
                <a href="training.php" class="nav-link">
                    <strong>📚 Train Your Bot</strong>
                    <span>Add FAQs and documents to your knowledge base</span>
                </a>
                <?php if ($vendor['plan_type'] === 'pro'): ?>
                <a href="active_chats.php" class="nav-link">
                    <strong>💬 Live Agent</strong>
                    <span>Manage active chat sessions and take over conversations</span>
                </a>
                <?php endif; ?>
            </div>
        </div>
    </div>
</body>
</html>
