/**
 * AI Chatbot Widget
 * Embeddable chat widget for customer websites
 */

(function() {
    'use strict';
    
    // Configuration
    const config = {
        vendorId: null,
        chatbotId: null,
        apiUrl: null,
        pusherKey: null,
        pusherCluster: null,
        sessionId: null,
        escalationAvailable: false
    };
    
    // State
    let widgetInitialized = false;
    let pusher = null;
    let channel = null;
    let isEscalated = false;
    
    /**
     * Initialize widget
     */
    function init(options) {
        if (widgetInitialized) {
            console.warn('Chatbot widget already initialized');
            return;
        }
        
        // Set configuration
        config.vendorId = options.vendorId || null;
        config.chatbotId = options.chatbotId || null;
        config.apiUrl = options.apiUrl || (window.location.origin + '/api/chat_gateway.php');
        config.pusherKey = options.pusherKey || null;
        config.pusherCluster = options.pusherCluster || 'us2';
        
        if (!config.vendorId || !config.chatbotId) {
            console.error('vendorId and chatbotId are required');
            return;
        }
        
        // Generate session ID
        config.sessionId = 'session_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
        
        // Create widget HTML
        createWidget();
        
        // Initialize Pusher if key is provided
        if (config.pusherKey && typeof Pusher !== 'undefined') {
            initPusher();
        }
        
        widgetInitialized = true;
    }
    
    /**
     * Create widget DOM elements
     */
    function createWidget() {
        // Create container
        const widget = document.createElement('div');
        widget.id = 'aichatbot-widget';
        document.body.appendChild(widget);
        
        // Create button
        const button = document.createElement('button');
        button.id = 'aichatbot-button';
        button.innerHTML = '💬';
        button.setAttribute('aria-label', 'Open chat');
        button.addEventListener('click', toggleWidget);
        widget.appendChild(button);
        
        // Create window
        const window = document.createElement('div');
        window.id = 'aichatbot-window';
        widget.appendChild(window);
        
        // Create header
        const header = document.createElement('div');
        header.id = 'aichatbot-header';
        header.innerHTML = `
            <h3>Chat Support</h3>
            <button id="aichatbot-close" aria-label="Close chat">×</button>
        `;
        window.appendChild(header);
        
        document.getElementById('aichatbot-close').addEventListener('click', toggleWidget);
        
        // Create messages container
        const messages = document.createElement('div');
        messages.id = 'aichatbot-messages';
        window.appendChild(messages);
        
        // Create input area
        const inputArea = document.createElement('div');
        inputArea.id = 'aichatbot-input-area';
        inputArea.innerHTML = `
            <button id="aichatbot-escalate-btn" style="display: none;">Talk to Human</button>
            <form id="aichatbot-input-form">
                <input 
                    type="text" 
                    id="aichatbot-input" 
                    placeholder="Type your message..." 
                    autocomplete="off"
                >
                <button type="submit" id="aichatbot-send">Send</button>
            </form>
        `;
        window.appendChild(inputArea);
        
        // Load CSS
        loadCSS();
        
        // Setup form handler
        document.getElementById('aichatbot-input-form').addEventListener('submit', handleSendMessage);
        
        // Setup escalate button
        document.getElementById('aichatbot-escalate-btn').addEventListener('click', handleEscalate);
    }
    
    /**
     * Load CSS file
     */
    function loadCSS() {
        const link = document.createElement('link');
        link.rel = 'stylesheet';
        link.href = config.apiUrl.replace('chat_gateway.php', '../widget/chatbot.css');
        document.head.appendChild(link);
    }
    
    /**
     * Initialize Pusher for real-time updates
     */
    function initPusher() {
        try {
            pusher = new Pusher(config.pusherKey, {
                cluster: config.pusherCluster
            });
            
            // Subscribe to vendor channel
            channel = pusher.subscribe('vendor-' + config.vendorId);
            
            // Listen for chat updates
            channel.bind('chat-update', function(data) {
                if (data.session_id === config.sessionId) {
                    if (data.role === 'agent') {
                        addMessage('agent', data.message);
                    }
                }
            });
            
            // Listen for escalation
            channel.bind('chat-escalated', function(data) {
                if (data.session_id === config.sessionId) {
                    isEscalated = true;
                    addMessage('bot', 'Your chat has been escalated to a live agent. Please wait for a response.');
                    document.getElementById('aichatbot-escalate-btn').style.display = 'none';
                }
            });
        } catch (error) {
            console.error('Pusher initialization error:', error);
        }
    }
    
    /**
     * Toggle widget open/close
     */
    function toggleWidget() {
        const window = document.getElementById('aichatbot-window');
        const button = document.getElementById('aichatbot-button');
        
        if (window.classList.contains('open')) {
            window.classList.remove('open');
            button.classList.remove('active');
        } else {
            window.classList.add('open');
            button.classList.add('active');
            document.getElementById('aichatbot-input').focus();
            
            // Show welcome message if first time
            if (document.getElementById('aichatbot-messages').children.length === 0) {
                fetchWelcomeMessage();
            }
        }
    }
    
    /**
     * Fetch welcome message
     */
    async function fetchWelcomeMessage() {
        try {
            const response = await fetch(config.apiUrl.replace('chat_gateway.php', '../api/get_chatbot_info.php'), {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    vendor_id: config.vendorId,
                    chatbot_id: config.chatbotId
                })
            });
            
            const data = await response.json();
            if (data.success && data.welcome_message) {
                addMessage('bot', data.welcome_message);
            }
        } catch (error) {
            console.error('Error fetching welcome message:', error);
        }
    }
    
    /**
     * Handle send message
     */
    async function handleSendMessage(e) {
        e.preventDefault();
        
        const input = document.getElementById('aichatbot-input');
        const message = input.value.trim();
        
        if (!message) {
            return;
        }
        
        // Disable input
        input.disabled = true;
        document.getElementById('aichatbot-send').disabled = true;
        
        // Add user message
        addMessage('user', message);
        input.value = '';
        
        // Show typing indicator
        showTyping();
        
        try {
            const response = await fetch(config.apiUrl, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    vendor_id: config.vendorId,
                    chatbot_id: config.chatbotId,
                    session_id: config.sessionId,
                    message: message
                })
            });
            
            const data = await response.json();
            hideTyping();
            
            if (data.success) {
                addMessage('bot', data.message);
                
                // Update escalation availability
                if (data.escalation_available) {
                    document.getElementById('aichatbot-escalate-btn').style.display = 'block';
                    config.escalationAvailable = true;
                }
            } else {
                addMessage('bot', 'Sorry, I encountered an error. Please try again.');
            }
        } catch (error) {
            hideTyping();
            addMessage('bot', 'Sorry, I encountered a network error. Please try again.');
            console.error('Chat error:', error);
        } finally {
            input.disabled = false;
            document.getElementById('aichatbot-send').disabled = false;
            input.focus();
        }
    }
    
    /**
     * Handle escalate to live agent
     */
    async function handleEscalate() {
        if (isEscalated) {
            return;
        }
        
        const btn = document.getElementById('aichatbot-escalate-btn');
        btn.disabled = true;
        btn.textContent = 'Escalating...';
        
        try {
            const response = await fetch(config.apiUrl.replace('chat_gateway.php', '../api/escalate_chat.php'), {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    vendor_id: config.vendorId,
                    chatbot_id: config.chatbotId,
                    session_id: config.sessionId
                })
            });
            
            const data = await response.json();
            
            if (data.success) {
                isEscalated = true;
                btn.style.display = 'none';
                addMessage('bot', 'Your chat has been escalated to a live agent. Please wait for a response.');
            } else {
                btn.disabled = false;
                btn.textContent = 'Talk to Human';
                alert('Failed to escalate chat. Please try again.');
            }
        } catch (error) {
            btn.disabled = false;
            btn.textContent = 'Talk to Human';
            alert('Network error. Please try again.');
            console.error('Escalate error:', error);
        }
    }
    
    /**
     * Add message to chat
     */
    function addMessage(role, message) {
        const messages = document.getElementById('aichatbot-messages');
        const messageDiv = document.createElement('div');
        messageDiv.className = 'aichatbot-message ' + role;
        
        const bubble = document.createElement('div');
        bubble.className = 'aichatbot-message-bubble';
        bubble.textContent = message;
        messageDiv.appendChild(bubble);
        
        const time = document.createElement('div');
        time.className = 'aichatbot-message-time';
        time.textContent = new Date().toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
        messageDiv.appendChild(time);
        
        messages.appendChild(messageDiv);
        messages.scrollTop = messages.scrollHeight;
    }
    
    /**
     * Show typing indicator
     */
    function showTyping() {
        const messages = document.getElementById('aichatbot-messages');
        const typing = document.createElement('div');
        typing.id = 'aichatbot-typing';
        typing.className = 'aichatbot-message bot';
        typing.innerHTML = `
            <div class="aichatbot-typing">
                <span></span>
                <span></span>
                <span></span>
            </div>
        `;
        messages.appendChild(typing);
        messages.scrollTop = messages.scrollHeight;
    }
    
    /**
     * Hide typing indicator
     */
    function hideTyping() {
        const typing = document.getElementById('aichatbot-typing');
        if (typing) {
            typing.remove();
        }
    }
    
    // Expose init function globally
    window.AIChatbotWidget = {
        init: init
    };
    
    // Auto-initialize if config is provided via data attributes
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', function() {
            const script = document.querySelector('script[data-vendor-id]');
            if (script) {
                init({
                    vendorId: script.getAttribute('data-vendor-id'),
                    chatbotId: script.getAttribute('data-chatbot-id'),
                    apiUrl: script.getAttribute('data-api-url'),
                    pusherKey: script.getAttribute('data-pusher-key'),
                    pusherCluster: script.getAttribute('data-pusher-cluster')
                });
            }
        });
    } else {
        const script = document.querySelector('script[data-vendor-id]');
        if (script) {
            init({
                vendorId: script.getAttribute('data-vendor-id'),
                chatbotId: script.getAttribute('data-chatbot-id'),
                apiUrl: script.getAttribute('data-api-url'),
                pusherKey: script.getAttribute('data-pusher-key'),
                pusherCluster: script.getAttribute('data-pusher-cluster')
            });
        }
    }
})();
